--
-- This file contains procedures and packages necessary to create local security
-- policies for an Oracle virtual private database
--

--
-- create SCOTT_SECURITY_POLICIES package and package body
--

create or replace package scott_security_policies as
   function emp_security(obj_schema varchar2, obj_name varchar2)
     return varchar2;
end scott_security_policies;
/

create or replace package body scott_security_policies as

 function emp_security (obj_schema varchar2, obj_name varchar2)
 return varchar2 is dynamic_predicate varchar2(2000);
    begin
     If (sys_context('userenv','session_user') = 'SCOTT') then
         dynamic_predicate := '';
     Elsif (sys_context('userenv','session_user') = 'DEMO') then
         dynamic_predicate := 'deptno=20';
     Else
         dynamic_predicate := '1=2';
     End if;
     return dynamic_predicate;
 end emp_security;

end scott_security_policies;
/

--
-- Add the security policy for the SCOTT.EMP table
--

exec dbms_rls.drop_policy('scott','emp','emp_policy')
/

exec dbms_rls.add_policy ('SCOTT', 'EMP', 'EMP_POLICY',
     'SCOTT', 'SCOTT_SECURITY_POLICIES.EMP_SECURITY',
     'SELECT, INSERT', TRUE, TRUE)
/


--
-- create the security context package that will be used to establish the
-- context and thus the security for the user.
--

CREATE OR REPLACE PACKAGE scott_context_pkg IS
   PROCEDURE set_deptno;
END;
/

CREATE OR REPLACE PACKAGE BODY scott_context_pkg IS 

   PROCEDURE set_deptno
   IS
   dept_no NUMBER;
   BEGIN
    SELECT DEPTNO INTO dept_no FROM SCOTT.DEPT 
       WHERE LOC = NEW YORK;

    DBMS_SESSION.SET_CONTEXT('scott_context', 'deptno', dept_no);

   EXCEPTION
     WHEN OTHERS THEN NULL;
   END set_deptno;

END scott_context_pkg;
/


CREATE OR REPLACE CONTEXT scott_context USING scott_context_pkg
;

-- 
-- create the updated security policies
--

create or replace package body scott_security_policies as

 function emp_security (obj_schema varchar2, obj_name varchar2)
 return varchar2 is dynamic_predicate varchar2(2000);
    begin
     If (sys_context('userenv','session_user') = 'SCOTT') then
         dynamic_predicate := '';
     Else
         dynamic_predicate := 
           'DEPTNO = SYS_CONTEXT(''scott_context'', ''deptno'')';
     End if;
     return dynamic_predicate;
 end emp_security;

end scott_security_policies;
/

exec DBMS_RLS.DROP_POLICY('scott','emp','emp_policy')
/

exec DBMS_RLS.ADD_POLICY ('SCOTT', 'EMP', 'EMP_POLICY',
       'SCOTT', 'SCOTT_SECURITY_POLICIES.EMP_SECURITY',
       'SELECT', TRUE, TRUE)
/

--
-- Finally, create a global login trigger that will call the context package for
-- each user connecting to the database. 
--

CREATE OR REPLACE TRIGGER scott_login_trigger
AFTER LOGON ON DATABASE
CALL scott.scott_context_pkg.set_deptno
/



